import { useState, useEffect } from "react";
import { useNavigate } from "react-router-dom";
import { Lock, CreditCard, Smartphone } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";
import { getCart, getCurrentUser, clearCart } from "@/lib/storage";
import { calculatePriceBreakdown, getCartTotal } from "@/lib/cart";
import { saveOrder } from "@/lib/storage";
import { kenyaCounties } from "@/data/mockProducts";
import { CartItem, ShippingAddress, Order } from "@/types/product";
import { toast } from "sonner";

const Checkout = () => {
  const navigate = useNavigate();
  const user = getCurrentUser();
  const [cart, setCart] = useState<CartItem[]>([]);
  const [isProcessing, setIsProcessing] = useState(false);

  // Shipping Form
  const [shippingAddress, setShippingAddress] = useState<ShippingAddress>({
    fullName: user?.name || "",
    phone: user?.phone || "",
    county: "",
    city: "",
    address: "",
    postalCode: "",
  });

  const [shippingMethod, setShippingMethod] = useState("standard");
  const [paymentMethod, setPaymentMethod] = useState("mpesa");

  useEffect(() => {
    const currentCart = getCart();
    if (currentCart.length === 0) {
      navigate("/cart");
    }
    setCart(currentCart);
  }, [navigate]);

  const shippingCosts = {
    standard: getCartTotal(cart) >= 5000 ? 0 : 300,
    express: 600,
  };

  const pricing = calculatePriceBreakdown(
    cart,
    shippingCosts[shippingMethod as keyof typeof shippingCosts]
  );

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();

    if (!user) {
      toast.error("Please login to place an order");
      navigate("/login");
      return;
    }

    // Validation
    if (!shippingAddress.fullName || !shippingAddress.phone || !shippingAddress.county || 
        !shippingAddress.city || !shippingAddress.address) {
      toast.error("Please fill in all required shipping information");
      return;
    }

    // Phone validation for Kenya
    const phoneRegex = /^(\+254|254|0)?[17]\d{8}$/;
    if (!phoneRegex.test(shippingAddress.phone)) {
      toast.error("Please enter a valid Kenyan phone number (e.g., +254712345678)");
      return;
    }

    setIsProcessing(true);

    // Simulate payment processing
    setTimeout(() => {
      const order: Order = {
        id: `ORD-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`,
        userId: user.id,
        items: cart,
        shippingAddress,
        subtotal: pricing.subtotal,
        vat: pricing.vat,
        shipping: pricing.shipping,
        total: pricing.total,
        paymentMethod,
        status: "pending",
        createdAt: new Date().toISOString(),
      };

      saveOrder(order);
      clearCart();
      
      toast.success("Order placed successfully!");
      navigate(`/order-confirmation/${order.id}`);
      setIsProcessing(false);
    }, 2000);
  };

  if (cart.length === 0) {
    return null;
  }

  return (
    <div className="min-h-screen bg-secondary/30">
      <div className="container mx-auto px-4 py-8">
        <h1 className="text-3xl font-bold mb-8">Checkout</h1>

        <form onSubmit={handleSubmit}>
          <div className="grid lg:grid-cols-3 gap-8">
            {/* Forms */}
            <div className="lg:col-span-2 space-y-6">
              {/* Shipping Address */}
              <Card className="p-6 gradient-card border-border">
                <h2 className="text-xl font-bold mb-6">Shipping Address</h2>
                <div className="grid md:grid-cols-2 gap-4">
                  <div className="md:col-span-2">
                    <Label htmlFor="fullName">Full Name *</Label>
                    <Input
                      id="fullName"
                      value={shippingAddress.fullName}
                      onChange={(e) => setShippingAddress({ ...shippingAddress, fullName: e.target.value })}
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="phone">Phone Number *</Label>
                    <Input
                      id="phone"
                      type="tel"
                      placeholder="+254712345678"
                      value={shippingAddress.phone}
                      onChange={(e) => setShippingAddress({ ...shippingAddress, phone: e.target.value })}
                      required
                    />
                    <p className="text-xs text-muted-foreground mt-1">Format: +254712345678</p>
                  </div>
                  <div>
                    <Label htmlFor="county">County *</Label>
                    <Select
                      value={shippingAddress.county}
                      onValueChange={(value) => setShippingAddress({ ...shippingAddress, county: value })}
                      required
                    >
                      <SelectTrigger id="county">
                        <SelectValue placeholder="Select county" />
                      </SelectTrigger>
                      <SelectContent>
                        {kenyaCounties.map(county => (
                          <SelectItem key={county} value={county}>
                            {county}
                          </SelectItem>
                        ))}
                      </SelectContent>
                    </Select>
                  </div>
                  <div>
                    <Label htmlFor="city">Town/City *</Label>
                    <Input
                      id="city"
                      value={shippingAddress.city}
                      onChange={(e) => setShippingAddress({ ...shippingAddress, city: e.target.value })}
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="address">Street Address *</Label>
                    <Input
                      id="address"
                      value={shippingAddress.address}
                      onChange={(e) => setShippingAddress({ ...shippingAddress, address: e.target.value })}
                      required
                    />
                  </div>
                  <div>
                    <Label htmlFor="postalCode">Postal Code</Label>
                    <Input
                      id="postalCode"
                      value={shippingAddress.postalCode}
                      onChange={(e) => setShippingAddress({ ...shippingAddress, postalCode: e.target.value })}
                    />
                  </div>
                </div>
              </Card>

              {/* Shipping Method */}
              <Card className="p-6 gradient-card border-border">
                <h2 className="text-xl font-bold mb-6">Shipping Method</h2>
                <RadioGroup value={shippingMethod} onValueChange={setShippingMethod}>
                  <div className="flex items-center justify-between p-4 border border-border rounded-lg mb-3">
                    <div className="flex items-center gap-3">
                      <RadioGroupItem value="standard" id="standard" />
                      <Label htmlFor="standard" className="cursor-pointer">
                        <p className="font-medium">Standard Delivery</p>
                        <p className="text-sm text-muted-foreground">2-5 business days</p>
                      </Label>
                    </div>
                    <span className="font-semibold">
                      {shippingCosts.standard === 0 ? "FREE" : `KSh ${shippingCosts.standard}`}
                    </span>
                  </div>
                  <div className="flex items-center justify-between p-4 border border-border rounded-lg">
                    <div className="flex items-center gap-3">
                      <RadioGroupItem value="express" id="express" />
                      <Label htmlFor="express" className="cursor-pointer">
                        <p className="font-medium">Express Delivery</p>
                        <p className="text-sm text-muted-foreground">1-2 business days</p>
                      </Label>
                    </div>
                    <span className="font-semibold">KSh {shippingCosts.express}</span>
                  </div>
                </RadioGroup>
              </Card>

              {/* Payment Method */}
              <Card className="p-6 gradient-card border-border">
                <h2 className="text-xl font-bold mb-6">Payment Method</h2>
                <RadioGroup value={paymentMethod} onValueChange={setPaymentMethod}>
                  <div className="flex items-center gap-3 p-4 border border-border rounded-lg mb-3">
                    <RadioGroupItem value="mpesa" id="mpesa" />
                    <Label htmlFor="mpesa" className="cursor-pointer flex items-center gap-3 flex-1">
                      <Smartphone className="h-5 w-5 text-success" />
                      <div>
                        <p className="font-medium">M-Pesa STK Push</p>
                        <p className="text-sm text-muted-foreground">Pay securely with M-Pesa</p>
                      </div>
                    </Label>
                  </div>
                  <div className="flex items-center gap-3 p-4 border border-border rounded-lg mb-3">
                    <RadioGroupItem value="card" id="card" />
                    <Label htmlFor="card" className="cursor-pointer flex items-center gap-3 flex-1">
                      <CreditCard className="h-5 w-5 text-primary" />
                      <div>
                        <p className="font-medium">Debit/Credit Card</p>
                        <p className="text-sm text-muted-foreground">Visa, Mastercard accepted</p>
                      </div>
                    </Label>
                  </div>
                  <div className="flex items-center gap-3 p-4 border border-border rounded-lg">
                    <RadioGroupItem value="cod" id="cod" />
                    <Label htmlFor="cod" className="cursor-pointer flex-1">
                      <p className="font-medium">Pay on Delivery</p>
                      <p className="text-sm text-muted-foreground">Cash payment when you receive</p>
                    </Label>
                  </div>
                </RadioGroup>
              </Card>
            </div>

            {/* Order Summary */}
            <div className="lg:col-span-1">
              <Card className="p-6 gradient-card border-border sticky top-24">
                <h2 className="text-xl font-bold mb-6">Order Summary</h2>
                
                {/* Items */}
                <div className="space-y-3 mb-6 max-h-60 overflow-y-auto">
                  {cart.map((item, idx) => (
                    <div key={idx} className="flex gap-3">
                      <img
                        src={item.product.images[0]}
                        alt={item.product.name}
                        className="w-16 h-16 object-cover rounded border border-border"
                      />
                      <div className="flex-1 min-w-0">
                        <p className="font-medium text-sm line-clamp-1">{item.product.name}</p>
                        <p className="text-xs text-muted-foreground">Qty: {item.quantity}</p>
                        {item.selectedColor && (
                          <p className="text-xs text-muted-foreground">{item.selectedColor}</p>
                        )}
                      </div>
                      <p className="text-sm font-medium">
                        KSh {(item.product.price * item.quantity).toLocaleString()}
                      </p>
                    </div>
                  ))}
                </div>

                {/* Price Breakdown */}
                <div className="space-y-2 mb-6 pb-6 border-b border-border">
                  <div className="flex justify-between text-sm">
                    <span className="text-muted-foreground">Subtotal</span>
                    <span>KSh {pricing.subtotal.toLocaleString()}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span className="text-muted-foreground">Shipping</span>
                    <span>{pricing.shipping === 0 ? "FREE" : `KSh ${pricing.shipping.toLocaleString()}`}</span>
                  </div>
                  <div className="flex justify-between text-sm">
                    <span className="text-muted-foreground">VAT (16%)</span>
                    <span>KSh {pricing.vat.toLocaleString()}</span>
                  </div>
                </div>

                <div className="flex justify-between items-baseline mb-6">
                  <span className="text-lg font-bold">Total</span>
                  <span className="text-2xl font-bold text-primary">
                    KSh {pricing.total.toLocaleString()}
                  </span>
                </div>

                <Button
                  type="submit"
                  size="lg"
                  className="w-full"
                  disabled={isProcessing}
                >
                  {isProcessing ? (
                    "Processing..."
                  ) : (
                    <>
                      <Lock className="h-4 w-4 mr-2" />
                      Place Order
                    </>
                  )}
                </Button>

                <p className="text-xs text-center text-muted-foreground mt-4">
                  Your payment information is secure and encrypted
                </p>
              </Card>
            </div>
          </div>
        </form>
      </div>
    </div>
  );
};

export default Checkout;
