import { useState, useEffect } from "react";
import { useNavigate, Link } from "react-router-dom";
import { Trash2, Plus, Minus, ShoppingBag } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { getCart } from "@/lib/storage";
import { updateCartItemQuantity, removeFromCart, getCartTotal } from "@/lib/cart";
import { CartItem } from "@/types/product";

const Cart = () => {
  const navigate = useNavigate();
  const [cart, setCart] = useState<CartItem[]>([]);

  useEffect(() => {
    setCart(getCart());
  }, []);

  const handleUpdateQuantity = (
    productId: string,
    newQuantity: number,
    selectedColor?: string,
    selectedSize?: string
  ) => {
    const updatedCart = updateCartItemQuantity(productId, newQuantity, selectedColor, selectedSize);
    setCart(updatedCart);
  };

  const handleRemove = (
    productId: string,
    selectedColor?: string,
    selectedSize?: string
  ) => {
    const updatedCart = removeFromCart(productId, selectedColor, selectedSize);
    setCart(updatedCart);
  };

  const subtotal = getCartTotal(cart);

  if (cart.length === 0) {
    return (
      <div className="container mx-auto px-4 py-16 text-center">
        <ShoppingBag className="h-24 w-24 mx-auto text-muted-foreground mb-6" />
        <h1 className="text-3xl font-bold mb-4">Your Cart is Empty</h1>
        <p className="text-muted-foreground mb-8">
          Add some products to get started!
        </p>
        <Button asChild size="lg">
          <Link to="/products">
            Continue Shopping
          </Link>
        </Button>
      </div>
    );
  }

  return (
    <div className="min-h-screen bg-secondary/30">
      <div className="container mx-auto px-4 py-8">
        <h1 className="text-3xl font-bold mb-8">Shopping Cart</h1>

        <div className="grid lg:grid-cols-3 gap-8">
          {/* Cart Items */}
          <div className="lg:col-span-2 space-y-4">
            {cart.map((item, index) => (
              <Card key={`${item.product.id}-${item.selectedColor}-${item.selectedSize}-${index}`} className="p-6 gradient-card border-border">
                <div className="flex gap-6">
                  {/* Image */}
                  <Link to={`/product/${item.product.id}`} className="shrink-0">
                    <img
                      src={item.product.images[0]}
                      alt={item.product.name}
                      className="w-24 h-24 object-cover rounded-lg border border-border"
                    />
                  </Link>

                  {/* Details */}
                  <div className="flex-1">
                    <Link to={`/product/${item.product.id}`}>
                      <h3 className="font-semibold text-lg mb-2 hover:text-primary transition-base">
                        {item.product.name}
                      </h3>
                    </Link>
                    
                    <div className="space-y-1 text-sm text-muted-foreground mb-3">
                      {item.selectedColor && (
                        <p>Color: <span className="font-medium text-foreground">{item.selectedColor}</span></p>
                      )}
                      {item.selectedSize && (
                        <p>Size: <span className="font-medium text-foreground">{item.selectedSize}</span></p>
                      )}
                    </div>

                    <p className="text-lg font-bold text-primary mb-4">
                      KSh {item.product.price.toLocaleString()}
                    </p>

                    {/* Quantity Controls */}
                    <div className="flex items-center gap-6">
                      <div className="flex items-center gap-2">
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() =>
                            handleUpdateQuantity(
                              item.product.id,
                              Math.max(1, item.quantity - 1),
                              item.selectedColor,
                              item.selectedSize
                            )
                          }
                          disabled={item.quantity <= 1}
                        >
                          <Minus className="h-3 w-3" />
                        </Button>
                        <span className="font-medium w-8 text-center">{item.quantity}</span>
                        <Button
                          variant="outline"
                          size="icon"
                          className="h-8 w-8"
                          onClick={() =>
                            handleUpdateQuantity(
                              item.product.id,
                              Math.min(item.product.stock, item.quantity + 1),
                              item.selectedColor,
                              item.selectedSize
                            )
                          }
                          disabled={item.quantity >= item.product.stock}
                        >
                          <Plus className="h-3 w-3" />
                        </Button>
                      </div>

                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => handleRemove(item.product.id, item.selectedColor, item.selectedSize)}
                        className="text-destructive hover:text-destructive"
                      >
                        <Trash2 className="h-4 w-4 mr-2" />
                        Remove
                      </Button>
                    </div>
                  </div>

                  {/* Item Total */}
                  <div className="text-right shrink-0">
                    <p className="text-sm text-muted-foreground mb-1">Subtotal</p>
                    <p className="text-xl font-bold">
                      KSh {(item.product.price * item.quantity).toLocaleString()}
                    </p>
                  </div>
                </div>
              </Card>
            ))}
          </div>

          {/* Order Summary */}
          <div className="lg:col-span-1">
            <Card className="p-6 gradient-card border-border sticky top-24">
              <h2 className="text-xl font-bold mb-6">Order Summary</h2>
              
              <div className="space-y-3 mb-6">
                <div className="flex justify-between">
                  <span className="text-muted-foreground">Subtotal</span>
                  <span className="font-medium">KSh {subtotal.toLocaleString()}</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">Shipping</span>
                  <span className="text-muted-foreground">Calculated at checkout</span>
                </div>
                <div className="flex justify-between text-sm">
                  <span className="text-muted-foreground">VAT (16%)</span>
                  <span className="text-muted-foreground">Calculated at checkout</span>
                </div>
              </div>

              <div className="border-t border-border pt-4 mb-6">
                <div className="flex justify-between items-baseline">
                  <span className="text-lg font-semibold">Estimated Total</span>
                  <span className="text-2xl font-bold text-primary">
                    KSh {subtotal.toLocaleString()}
                  </span>
                </div>
                <p className="text-xs text-muted-foreground mt-1">
                  Final total will be calculated at checkout
                </p>
              </div>

              <Button
                onClick={() => navigate("/checkout")}
                size="lg"
                className="w-full mb-3"
              >
                Proceed to Checkout
              </Button>

              <Button
                asChild
                variant="outline"
                size="lg"
                className="w-full"
              >
                <Link to="/products">
                  Continue Shopping
                </Link>
              </Button>

              {subtotal >= 5000 && (
                <div className="mt-4 p-3 bg-success/10 border border-success/20 rounded-lg text-sm text-center">
                  🎉 You qualify for free shipping!
                </div>
              )}
            </Card>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Cart;
