import { useEffect } from "react";
import { useNavigate, Link } from "react-router-dom";
import { User, Package, LogOut } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Card } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { getCurrentUser, getUserOrders, logout } from "@/lib/storage";
import { toast } from "sonner";

const Account = () => {
  const navigate = useNavigate();
  const user = getCurrentUser();
  const orders = user ? getUserOrders(user.id) : [];

  useEffect(() => {
    if (!user) {
      navigate("/login");
    }
  }, [user, navigate]);

  const handleLogout = () => {
    logout();
    toast.success("Logged out successfully");
    navigate("/");
  };

  if (!user) return null;

  const statusColors = {
    pending: "bg-warning text-warning-foreground",
    processing: "bg-primary text-primary-foreground",
    shipped: "bg-primary text-primary-foreground",
    delivered: "bg-success text-success-foreground",
    cancelled: "bg-destructive text-destructive-foreground",
  };

  return (
    <div className="min-h-screen bg-secondary/30">
      <div className="container mx-auto px-4 py-8">
        <div className="grid lg:grid-cols-4 gap-8">
          {/* Sidebar */}
          <div className="lg:col-span-1">
            <Card className="p-6 gradient-card border-border">
              <div className="flex items-center gap-3 mb-6 pb-6 border-b border-border">
                <div className="w-12 h-12 gradient-hero rounded-full flex items-center justify-center text-primary-foreground font-bold text-lg">
                  {user.name.charAt(0).toUpperCase()}
                </div>
                <div>
                  <p className="font-semibold">{user.name}</p>
                  <p className="text-sm text-muted-foreground">{user.email}</p>
                </div>
              </div>

              <nav className="space-y-2">
                <Button variant="ghost" className="w-full justify-start" asChild>
                  <a href="#profile">
                    <User className="h-4 w-4 mr-2" />
                    Profile
                  </a>
                </Button>
                <Button variant="default" className="w-full justify-start" asChild>
                  <a href="#orders">
                    <Package className="h-4 w-4 mr-2" />
                    My Orders
                  </a>
                </Button>
                <Button
                  variant="ghost"
                  className="w-full justify-start text-destructive hover:text-destructive"
                  onClick={handleLogout}
                >
                  <LogOut className="h-4 w-4 mr-2" />
                  Logout
                </Button>
              </nav>
            </Card>
          </div>

          {/* Main Content */}
          <div className="lg:col-span-3 space-y-8">
            {/* Profile Section */}
            <section id="profile">
              <h1 className="text-3xl font-bold mb-6">My Account</h1>
              <Card className="p-6 gradient-card border-border">
                <h2 className="text-xl font-bold mb-6">Profile Information</h2>
                <div className="grid md:grid-cols-2 gap-6">
                  <div>
                    <p className="text-sm text-muted-foreground mb-1">Full Name</p>
                    <p className="font-medium">{user.name}</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground mb-1">Email</p>
                    <p className="font-medium">{user.email}</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground mb-1">Phone Number</p>
                    <p className="font-medium">{user.phone}</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground mb-1">Account ID</p>
                    <p className="font-medium text-xs">{user.id}</p>
                  </div>
                </div>
              </Card>
            </section>

            {/* Orders Section */}
            <section id="orders">
              <h2 className="text-2xl font-bold mb-6">Order History</h2>
              {orders.length === 0 ? (
                <Card className="p-12 text-center gradient-card border-border">
                  <Package className="h-16 w-16 mx-auto text-muted-foreground mb-4" />
                  <h3 className="text-xl font-semibold mb-2">No Orders Yet</h3>
                  <p className="text-muted-foreground mb-6">
                    Start shopping to see your orders here
                  </p>
                  <Button asChild>
                    <Link to="/products">Browse Products</Link>
                  </Button>
                </Card>
              ) : (
                <div className="space-y-4">
                  {orders.map((order) => (
                    <Card key={order.id} className="p-6 gradient-card border-border">
                      <div className="flex flex-col md:flex-row md:items-center justify-between gap-4 mb-4">
                        <div>
                          <h3 className="font-bold text-lg mb-1">Order #{order.id}</h3>
                          <p className="text-sm text-muted-foreground">
                            {new Date(order.createdAt).toLocaleDateString("en-KE", {
                              year: "numeric",
                              month: "long",
                              day: "numeric",
                            })}
                          </p>
                        </div>
                        <Badge className={statusColors[order.status]}>
                          {order.status.charAt(0).toUpperCase() + order.status.slice(1)}
                        </Badge>
                      </div>

                      <div className="space-y-3 mb-4">
                        {order.items.map((item, idx) => (
                          <div key={idx} className="flex gap-4">
                            <img
                              src={item.product.images[0]}
                              alt={item.product.name}
                              className="w-16 h-16 object-cover rounded border border-border"
                            />
                            <div className="flex-1">
                              <p className="font-medium">{item.product.name}</p>
                              <p className="text-sm text-muted-foreground">
                                Qty: {item.quantity} × KSh {item.product.price.toLocaleString()}
                              </p>
                              {item.selectedColor && (
                                <p className="text-xs text-muted-foreground">Color: {item.selectedColor}</p>
                              )}
                              {item.selectedSize && (
                                <p className="text-xs text-muted-foreground">Size: {item.selectedSize}</p>
                              )}
                            </div>
                            <div className="text-right">
                              <p className="font-semibold">
                                KSh {(item.product.price * item.quantity).toLocaleString()}
                              </p>
                            </div>
                          </div>
                        ))}
                      </div>

                      <div className="border-t border-border pt-4">
                        <div className="flex justify-between items-center">
                          <div className="text-sm space-y-1">
                            <p className="text-muted-foreground">
                              Shipping: {order.shippingAddress.city}, {order.shippingAddress.county}
                            </p>
                            <p className="text-muted-foreground">Payment: {order.paymentMethod}</p>
                          </div>
                          <div className="text-right">
                            <p className="text-sm text-muted-foreground mb-1">Order Total</p>
                            <p className="text-2xl font-bold text-primary">
                              KSh {order.total.toLocaleString()}
                            </p>
                          </div>
                        </div>
                      </div>
                    </Card>
                  ))}
                </div>
              )}
            </section>
          </div>
        </div>
      </div>
    </div>
  );
};

export default Account;
