import { CartItem, Product } from "@/types/product";
import { getCart, saveCart } from "./storage";

export const addToCart = (
  product: Product,
  quantity: number = 1,
  selectedColor?: string,
  selectedSize?: string
): CartItem[] => {
  const cart = getCart();
  
  // Check if item already exists with same options
  const existingIndex = cart.findIndex(
    item =>
      item.product.id === product.id &&
      item.selectedColor === selectedColor &&
      item.selectedSize === selectedSize
  );

  if (existingIndex >= 0) {
    // Update quantity (ensure not exceeding stock)
    const newQuantity = Math.min(
      cart[existingIndex].quantity + quantity,
      product.stock
    );
    cart[existingIndex].quantity = newQuantity;
  } else {
    // Add new item
    cart.push({
      product,
      quantity: Math.min(quantity, product.stock),
      selectedColor,
      selectedSize,
    });
  }

  saveCart(cart);
  return cart;
};

export const updateCartItemQuantity = (
  productId: string,
  quantity: number,
  selectedColor?: string,
  selectedSize?: string
): CartItem[] => {
  const cart = getCart();
  const itemIndex = cart.findIndex(
    item =>
      item.product.id === productId &&
      item.selectedColor === selectedColor &&
      item.selectedSize === selectedSize
  );

  if (itemIndex >= 0) {
    if (quantity <= 0) {
      cart.splice(itemIndex, 1);
    } else {
      cart[itemIndex].quantity = Math.min(quantity, cart[itemIndex].product.stock);
    }
  }

  saveCart(cart);
  return cart;
};

export const removeFromCart = (
  productId: string,
  selectedColor?: string,
  selectedSize?: string
): CartItem[] => {
  let cart = getCart();
  cart = cart.filter(
    item =>
      !(
        item.product.id === productId &&
        item.selectedColor === selectedColor &&
        item.selectedSize === selectedSize
      )
  );

  saveCart(cart);
  return cart;
};

export const getCartTotal = (cart: CartItem[]): number => {
  return cart.reduce((total, item) => total + item.product.price * item.quantity, 0);
};

export const getCartCount = (cart: CartItem[]): number => {
  return cart.reduce((count, item) => count + item.quantity, 0);
};

const VAT_RATE = 0.16; // 16% VAT in Kenya

export const calculatePriceBreakdown = (
  cart: CartItem[],
  shippingCost: number
): {
  subtotal: number;
  vat: number;
  shipping: number;
  total: number;
} => {
  const subtotal = getCartTotal(cart);
  const vat = subtotal * VAT_RATE;
  const total = subtotal + vat + shippingCost;

  return {
    subtotal,
    vat,
    shipping: shippingCost,
    total,
  };
};
