import { useState } from "react";
import { Link, useNavigate } from "react-router-dom";
import { ShoppingCart, User, Search, Menu, X } from "lucide-react";
import { Button } from "./ui/button";
import { Input } from "./ui/input";
import { Badge } from "./ui/badge";
import { getCart } from "@/lib/storage";
import { getCartCount } from "@/lib/cart";
import { getCurrentUser } from "@/lib/storage";

const Header = () => {
  const [mobileMenuOpen, setMobileMenuOpen] = useState(false);
  const [searchQuery, setSearchQuery] = useState("");
  const navigate = useNavigate();
  const user = getCurrentUser();
  const cartCount = getCartCount(getCart());

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault();
    if (searchQuery.trim()) {
      navigate(`/products?search=${encodeURIComponent(searchQuery.trim())}`);
      setMobileMenuOpen(false);
    }
  };

  return (
    <header className="sticky top-0 z-50 bg-card border-b border-border shadow-custom-sm">
      {/* Top Bar */}
      <div className="gradient-hero text-primary-foreground py-2">
        <div className="container mx-auto px-4 text-center text-sm">
          Free shipping on orders over KSh 5,000 | Shop with M-Pesa
        </div>
      </div>

      {/* Main Header */}
      <div className="container mx-auto px-4 py-4">
        <div className="flex items-center justify-between gap-4">
          {/* Logo */}
          <Link to="/" className="flex items-center gap-2">
            <div className="w-10 h-10 gradient-hero rounded-lg flex items-center justify-center font-bold text-primary-foreground text-xl">
              K
            </div>
            <span className="font-bold text-xl hidden sm:block">Kenya Shop</span>
          </Link>

          {/* Search Bar - Desktop */}
          <form onSubmit={handleSearch} className="hidden md:flex flex-1 max-w-xl">
            <div className="relative w-full">
              <Input
                type="search"
                placeholder="Search products..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="pr-10"
              />
              <Button
                type="submit"
                size="icon"
                variant="ghost"
                className="absolute right-0 top-0 h-full"
              >
                <Search className="h-4 w-4" />
              </Button>
            </div>
          </form>

          {/* Actions */}
          <div className="flex items-center gap-2">
            <Button
              variant="ghost"
              size="icon"
              onClick={() => navigate(user ? "/account" : "/login")}
              className="relative"
            >
              <User className="h-5 w-5" />
            </Button>

            <Button
              variant="ghost"
              size="icon"
              onClick={() => navigate("/cart")}
              className="relative"
            >
              <ShoppingCart className="h-5 w-5" />
              {cartCount > 0 && (
                <Badge className="absolute -top-1 -right-1 h-5 w-5 flex items-center justify-center p-0 bg-accent text-accent-foreground">
                  {cartCount}
                </Badge>
              )}
            </Button>

            <Button
              variant="ghost"
              size="icon"
              onClick={() => setMobileMenuOpen(!mobileMenuOpen)}
              className="md:hidden"
            >
              {mobileMenuOpen ? <X className="h-5 w-5" /> : <Menu className="h-5 w-5" />}
            </Button>
          </div>
        </div>

        {/* Search Bar - Mobile */}
        <form onSubmit={handleSearch} className="mt-4 md:hidden">
          <div className="relative">
            <Input
              type="search"
              placeholder="Search products..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="pr-10"
            />
            <Button
              type="submit"
              size="icon"
              variant="ghost"
              className="absolute right-0 top-0 h-full"
            >
              <Search className="h-4 w-4" />
            </Button>
          </div>
        </form>

        {/* Navigation - Desktop */}
        <nav className="hidden md:flex items-center gap-6 mt-4 border-t border-border pt-4">
          <Link to="/products" className="text-sm hover:text-primary transition-base">
            All Products
          </Link>
          <Link to="/products?category=electronics" className="text-sm hover:text-primary transition-base">
            Electronics
          </Link>
          <Link to="/products?category=clothing" className="text-sm hover:text-primary transition-base">
            Clothing
          </Link>
          <Link to="/products?category=home" className="text-sm hover:text-primary transition-base">
            Home
          </Link>
          <Link to="/products?category=beauty" className="text-sm hover:text-primary transition-base">
            Beauty
          </Link>
          <Link to="/products?category=fashion" className="text-sm hover:text-primary transition-base">
            Fashion
          </Link>
        </nav>
      </div>

      {/* Mobile Menu */}
      {mobileMenuOpen && (
        <div className="md:hidden border-t border-border bg-card">
          <nav className="container mx-auto px-4 py-4 flex flex-col gap-2">
            <Link
              to="/products"
              onClick={() => setMobileMenuOpen(false)}
              className="py-2 hover:text-primary transition-base"
            >
              All Products
            </Link>
            <Link
              to="/products?category=electronics"
              onClick={() => setMobileMenuOpen(false)}
              className="py-2 hover:text-primary transition-base"
            >
              Electronics
            </Link>
            <Link
              to="/products?category=clothing"
              onClick={() => setMobileMenuOpen(false)}
              className="py-2 hover:text-primary transition-base"
            >
              Clothing
            </Link>
            <Link
              to="/products?category=home"
              onClick={() => setMobileMenuOpen(false)}
              className="py-2 hover:text-primary transition-base"
            >
              Home
            </Link>
            <Link
              to="/products?category=beauty"
              onClick={() => setMobileMenuOpen(false)}
              className="py-2 hover:text-primary transition-base"
            >
              Beauty
            </Link>
            <Link
              to="/products?category=fashion"
              onClick={() => setMobileMenuOpen(false)}
              className="py-2 hover:text-primary transition-base"
            >
              Fashion
            </Link>
          </nav>
        </div>
      )}
    </header>
  );
};

export default Header;
